<?php
/**
 * Hero Slider Database Test
 * Verify the database table exists and is working correctly
 */

require_once 'config.php';
session_start();

$testResults = [];

// Test 1: Check if table exists
$tableCheck = $conn->query("SELECT * FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = '" . DB_NAME . "' AND TABLE_NAME = 'hero_slider'");

if ($tableCheck && $tableCheck->num_rows > 0) {
    $testResults['table_exists'] = ['status' => 'PASS', 'message' => 'hero_slider table exists'];
} else {
    $testResults['table_exists'] = ['status' => 'FAIL', 'message' => 'hero_slider table does not exist. Run create_hero_slider_table.sql'];
}

// Test 2: Check if data exists
$rowCount = 0;
$rowCheck = $conn->query("SELECT COUNT(*) as count FROM hero_slider");
if ($rowCheck) {
    $row = $rowCheck->fetch_assoc();
    $rowCount = $row['count'];
    $testResults['data_exists'] = ['status' => ($rowCount > 0 ? 'PASS' : 'WARN'), 'message' => "Found $rowCount slides in database" . ($rowCount == 0 ? ' (table is empty, add a slide via admin panel)' : '')];
}

// Test 3: Try to fetch slides
$slides = [];
$query = "SELECT id, title, badge, is_active FROM hero_slider LIMIT 5";
$result = $conn->query($query);
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $slides[] = $row;
    }
    $testResults['fetch_slides'] = ['status' => 'PASS', 'message' => 'Successfully queried slides'];
}

?><!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hero Slider Database Test</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #f8fbff, #f1f5f4);
            padding: 40px 20px;
            min-height: 100vh;
        }
        
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            padding: 40px;
        }
        
        h1 {
            color: #109c3d;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .subtitle {
            color: #64748b;
            margin-bottom: 30px;
            font-size: 14px;
        }
        
        .test-section {
            margin-bottom: 30px;
        }
        
        .test-item {
            background: #f8fafc;
            border-left: 4px solid #e2e8f0;
            padding: 15px;
            margin-bottom: 12px;
            border-radius: 6px;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .test-item.pass {
            border-left-color: #10b981;
            background: #f0fdf4;
        }
        
        .test-item.fail {
            border-left-color: #ef4444;
            background: #fef2f2;
        }
        
        .test-item.warn {
            border-left-color: #f59e0b;
            background: #fffbf0;
        }
        
        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 600;
            font-size: 12px;
            white-space: nowrap;
            min-width: 70px;
            text-align: center;
        }
        
        .status-badge.pass {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-badge.fail {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .status-badge.warn {
            background: #fef3c7;
            color: #92400e;
        }
        
        .message {
            flex: 1;
        }
        
        .message strong {
            display: block;
            color: #0f172a;
            margin-bottom: 4px;
        }
        
        .message p {
            color: #64748b;
            font-size: 13px;
            margin: 0;
        }
        
        .data-table {
            margin-top: 20px;
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table th {
            background: #f1f5f9;
            color: #0f172a;
            padding: 12px;
            text-align: left;
            font-weight: 600;
            border-bottom: 2px solid #e2e8f0;
        }
        
        .data-table td {
            padding: 12px;
            border-bottom: 1px solid #e2e8f0;
            color: #475569;
        }
        
        .data-table tr:hover {
            background: #f8fafc;
        }
        
        .action-buttons {
            margin-top: 30px;
            display: flex;
            gap: 10px;
        }
        
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn-primary {
            background: #109c3d;
            color: white;
        }
        
        .btn-primary:hover {
            background: #0f8c3d;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(16,156,61,0.3);
        }
        
        .btn-secondary {
            background: #e2e8f0;
            color: #0f172a;
        }
        
        .btn-secondary:hover {
            background: #cbd5e1;
        }
        
        .alert {
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .alert-info {
            background: #eff6ff;
            color: #0c4a6e;
            border-left: 4px solid #0284c7;
        }
        
        .icon {
            font-size: 18px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>
            <span class="icon">🔍</span>
            Hero Slider Database Test
        </h1>
        <p class="subtitle">Verify database connection and table setup</p>
        
        <div class="alert alert-info">
            <strong>ℹ️ Database Configuration</strong><br>
            Host: <?php echo DB_HOST; ?> | User: <?php echo DB_USER; ?> | Database: <?php echo DB_NAME; ?>
        </div>
        
        <div class="test-section">
            <h3 style="color: #0f172a; margin-bottom: 15px;">System Tests</h3>
            
            <?php foreach ($testResults as $testName => $result): ?>
                <div class="test-item <?php echo strtolower($result['status']); ?>">
                    <div class="status-badge <?php echo strtolower($result['status']); ?>">
                        <?php echo $result['status']; ?>
                    </div>
                    <div class="message">
                        <strong><?php echo str_replace('_', ' ', ucfirst($testName)); ?></strong>
                        <p><?php echo $result['message']; ?></p>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        
        <?php if (!empty($slides)): ?>
            <div class="test-section">
                <h3 style="color: #0f172a; margin-bottom: 15px;">Sample Data (First 5 Slides)</h3>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Title</th>
                            <th>Badge</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($slides as $slide): ?>
                            <tr>
                                <td><?php echo $slide['id']; ?></td>
                                <td><?php echo htmlspecialchars($slide['title']); ?></td>
                                <td><?php echo htmlspecialchars($slide['badge'] ?? '-'); ?></td>
                                <td>
                                    <?php 
                                        if ($slide['is_active']) {
                                            echo '<span style="color: #10b981; font-weight: 600;">Active</span>';
                                        } else {
                                            echo '<span style="color: #ef4444; font-weight: 600;">Inactive</span>';
                                        }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
        
        <div class="action-buttons">
            <a href="index.php" class="btn btn-primary">View Homepage</a>
            <a href="admin-dashboard.php" class="btn btn-primary">Admin Dashboard</a>
            <button class="btn btn-secondary" onclick="location.reload()">Refresh Test</button>
        </div>
    </div>
</body>
</html>
